/*=================================================================
 *
 * uncond_loss_dist3.c ... Calculate the unconditional distribution of the number of defaults in a
 *                         portfolio by numerical integration over the conditional distribution.
 *
 * The calling syntax is:
 *
 *		uncond_loss_dist3(ai, p_surv_idio, points, density, density_middle);
 *
 * ai               ... row vector of loadings on common factor
 * p_surv_idio      ... row vector of idiosyncratic survival probabilities
 * points           ... row vector with N support points of the density
 * density_middle   ... k x N matrix with density of k Gauss-Legendre points for each quantile interval
 *
 *      sample call: uncond_loss_dist3(ones(1,10), repmat(0.99, 1, 10), 0:0.01:0.1, [1:10; 21:30])
 *                   uncond_loss_dist3(ones(1,10), repmat(0.99, 1, 10), 0:0.01:0.1, repmat(10, 5, 10))
 *                   uncond_loss_dist3(ones(1,10), repmat(0.99, 1, 10), 0:0.001:0.1, repmat(10, 5, 100))
 *                   uncond_loss_dist3(ones(1,125), repmat(0.99, 1, 125), 0:0.0001:0.1, repmat(1, 5, 1000));
 *=================================================================*/

#include <math.h>
#include "mex.h"

#ifndef max
#define max(a,b) (((a) > (b)) ? (a) : (b))
#endif

/* Recursive computation of loss distribution */
static void cond_loss_c(double p[], double loss[], unsigned int num_firms)
{
    int i, j;
    double exp_def = 0;
    int threshold;
    
    /* Compute expected number of defaults */
    for (i=0; i<num_firms; i++) 
    {
        exp_def += p[i];
    }
     
    /* Calculate threshold K, which bounds the number of defaults with prob. 1-1e-14 */
    threshold = (int) (8 + exp_def + 8*sqrt(exp_def));
    threshold = -max(-num_firms, - threshold);
    
    loss[0] = 1;
    for (i=0; i<num_firms; i++) 
    {
        /* Clear memory from potentially previous run of function */
        loss[i+1] = 0;

        /* Update loss distribution when adding one asset */
        for (j=-max(-i-1,-threshold); j>=1; j--)
        {
            loss[j] = loss[j]*(1-p[i]) + loss[j-1]*p[i];
        }
        loss[0] = loss[0] * (1-p[i]);
    }
    /* loss[0] = (double) threshold; */
    return;
}

/* Wrapper function that calculates conditional default probabilities & does the numerical integration */
static void wrapper_loss_dist(double loss[], double ai[], double p_surv_idio[], double points[],
                              double density_middle[], unsigned int num_firms, unsigned int N, unsigned int GL_num_points)
{
    int i, j, k;
    double sum, tmp, integration_weight;
    double *p_surv_syst, *p_def, *cond_loss_dist, x, *GL_points, *GL_weights;
    
    /* Allocate memory for temporary variables */
    p_surv_syst = mxGetPr(mxCreateDoubleMatrix(num_firms, 1, mxREAL)); 
    p_def = mxGetPr(mxCreateDoubleMatrix(num_firms, 1, mxREAL)); 
    cond_loss_dist = mxGetPr(mxCreateDoubleMatrix(num_firms+1, 1, mxREAL)); 
    GL_points = mxGetPr(mxCreateDoubleMatrix(GL_num_points, 1, mxREAL)); 
    GL_weights = mxGetPr(mxCreateDoubleMatrix(GL_num_points, 1, mxREAL)); 
    
    /* Calculate Gauss-Legendre weights and support points */
    if (GL_num_points == 2) {
        /* Points */
        GL_points[0] = -1.0 / sqrt(3);
        GL_points[1] = -GL_points[0];
        /* Weights */
        GL_weights[0] = GL_weights[1] = 1.0;
    } else if (GL_num_points == 3) {
        /* Points */
        GL_points[0] = -sqrt(15.0)/5;
        GL_points[1] = 0;
        GL_points[2] = -GL_points[0];
        /* Weights */
        GL_weights[0] = GL_weights[2] = 5.0 / 9;
        GL_weights[1] = 8.0 /9;        
    } else if (GL_num_points == 4) {
        /* Points */
        GL_points[0] = -1.0 / 35 * sqrt(525+70*sqrt(30));
        GL_points[1] = -1.0 / 35 * sqrt(525-70*sqrt(30));
        GL_points[2] = -GL_points[1];
        GL_points[3] = -GL_points[0];
        /* Weights */
        GL_weights[0] = GL_weights[3] = 1.0 / 36 * (18 - sqrt(30));
        GL_weights[1] = GL_weights[2] = 1.0 / 36 * (18 + sqrt(30));
    } else if (GL_num_points == 5) {
        /* Points */
        GL_points[0] = -1.0 / 21 * sqrt(245+14*sqrt(70));
        GL_points[1] = -1.0 / 21 * sqrt(245-14*sqrt(70));
        GL_points[2] = 0;
        GL_points[3] = -GL_points[1];
        GL_points[4] = -GL_points[0];
        /* Weights */
        GL_weights[0] = GL_weights[4] = 1.0/900 * (322 - 13*sqrt(70));
        GL_weights[1] = GL_weights[3] = 1.0/900 * (322 + 13*sqrt(70));
        GL_weights[2] = 128.0/225;
    }
    
    /* Perform numerical integration (Gauss-Legendre in each interval) */
    for (k=0; k<GL_num_points; k++) {     
        for (j=1; j<N; j++)
        {
            /* Calculate current x-value of integration */
            x = points[j-1] + (1 + GL_points[k])/2 * (points[j] - points[j-1]);
            
            /* Calculate conditional default probabilities */
            for (i=0; i<num_firms; i++)
            {
                tmp = x * ai[i];
                if (tmp < 0.1) {
                    /* Use Taylor expansion instead of exp => faster */
                    p_surv_syst[i] = 1 - tmp + tmp*tmp/2 - tmp*tmp*tmp/6 + tmp*tmp*tmp*tmp/24;
                } else {
                    p_surv_syst[i] = exp( -x * ai[i]);
                }
                p_def[i] = 1 - p_surv_idio[i] * p_surv_syst[i];
            }

            /* Calculate conditional portfolio loss distribution */
            cond_loss_c(p_def, cond_loss_dist, num_firms);
            
            /* Calculate integration weight */
            integration_weight = GL_weights[k] / 2 * (points[j] - points[j-1]);

            /* Add conditional loss distribution to unconditional one (weighted by density) */
            tmp = integration_weight * density_middle[k + (j-1) * GL_num_points];
            for (i=0; i<num_firms+1; i++)
            {
                loss[i] += cond_loss_dist[i] * tmp;
            }
            /* loss[j-1] = 0; /*density_middle[0]; */
        }
        /* loss[k] = density_middle[0]; */
    }
    /* loss[0] = density_middle[1]; */
    
    /* Normalize distribution to sum to 1 */
    sum = 0;
    for (i=0; i<num_firms+1; i++)
    {
        sum += loss[i];
    }
    for (i=0; i<num_firms+1; i++)
    {
        loss[i] = loss[i] / sum;
    }
    
    return;
}


/* Gateway routine (to Matlab) */
void mexFunction( int nlhs, mxArray *plhs[], 
		  int nrhs, const mxArray*prhs[] )
     
{ 
    double *ai, *p_surv_idio, *points, *density_middle; 
    double *loss;
    unsigned int num_firms, N, GL_num_points; 

    /* Get number of companies, number of points for numerical integration */
    num_firms = mxGetN(prhs[1]);
    N = mxGetN(prhs[2]);
    GL_num_points = mxGetM(prhs[3]);
    
    /* Create a matrix for the return argument */ 
    plhs[0] = mxCreateDoubleMatrix(1, num_firms+1, mxREAL); 
    
    /* Assign pointers to the various parameters */ 
    loss = mxGetPr(plhs[0]);
    ai = mxGetPr(prhs[0]);
    p_surv_idio = mxGetPr(prhs[1]);
    points = mxGetPr(prhs[2]);
    density_middle = mxGetPr(prhs[3]);
    
    /* Do the actual computations in a subroutine */
    wrapper_loss_dist(loss, ai, p_surv_idio, points, density_middle, num_firms, N, GL_num_points);
    return;
}
